/*
 * Gnophone: A client for the Asterisk PBX
 *
 * Copyright (C) 2000, Linux Support Services, Inc.
 *
 * Written by Mark Spencer
 *
 * Linux/UNIX version distributed under the terms of
 * the GNU General Public License
 *
 * phonecore.c: Core telephony thread.  This is something of a wrapper
 * around libiax to keep the audio stuff separated into a high performancce
 * thread.  There really should be a better way to do this though.
 *
 */

#ifndef _PHONECORE_H
#define _PHONECORE_H

#ifndef WIN32
#include <netinet/in.h>
#else
#include <windows.h>
#include <winsock2.h>
#endif

#define PC_EVENT_IMAGE		1
#define PC_EVENT_AUDIO  	2
#define PC_EVENT_LOADCOMPLETE 3
#define PC_EVENT_DTMF		4
#define PC_EVENT_HANGUP		5
#define PC_EVENT_ANSWER		6
#define PC_EVENT_REJECT		7
#define PC_EVENT_ACCEPT	    8
#define PC_EVENT_AUTHRQ		9
#define PC_EVENT_AUTHRP		10
#define PC_EVENT_URL		11
#define PC_EVENT_RINGA		12
#define PC_EVENT_REGREP		13
#define PC_EVENT_CONNECT	14
#define PC_EVENT_SELECT		15
#define PC_EVENT_NEW		16
#define PC_EVENT_NEW_REPLY	17
#define PC_EVENT_SELAUDIO		18
#define PC_EVENT_AUDIO_REPLY	19
#define PC_EVENT_REGREQ		20
#define PC_EVENT_AUDIO_READY 21

#define PC_EVENT_AUDIO_DIGIT	22
#define PC_EVENT_AUDIO_RING		23
#define PC_EVENT_AUDIO_RINGING	24
#define PC_EVENT_AUDIO_SHUTUP	25
#define PC_EVENT_AUDIO_DEACTIVATE	26
#define PC_EVENT_ADDR 		27
#define PC_EVENT_CONFERENCE	28
#define PC_EVENT_UNCONFERENCE 29

#define PC_EVENT_ONHOOK 30
#define PC_EVENT_OFFHOOK 31

#define PC_EVENT_DIAL 32

#define PC_EVENT_DIAL_TBD 33
#define PC_EVENT_DPREQ 34
#define PC_EVENT_DPREP 35
#define PC_EVENT_COMPLETE 36
#define PC_EVENT_TRANSFER 37
#define PC_EVENT_LAGREQ 38
#define PC_EVENT_LAGREP 39
#define PC_EVENT_UNLINK 40
#define PC_EVENT_LINKREJECT 41
#define PC_EVENT_TEXT	42

#define PC_REG_SUCCESS	1
#define PC_REG_REJECT	2
#define PC_REG_TIMEOUT	3

/* No more than 32 calls simulataneously */
#define PC_MAX_CALLS		32

struct text_event {
	unsigned char text[8192];
};

struct image_event {
	int datalen;

	/* No images more than 16k */
	unsigned char data[8192];
};

struct audio_event {
	/* Audio level change */
	int level;
};

struct dpreq_event {
	unsigned char dest[256];
};

struct dprep_event {
	unsigned char number[256];
	int exists;
	int canexist;
	int nonexistant;
	int ignorepat;
};

struct lagrep_event {
	int lag;
	int jitter;
};

struct dtmf_event {
	char dtmf;
};

struct dial_event {
	char digit;
};

struct hangup_event {
	char why[256];
};

struct reject_event {
	char why[256];
};

struct connect_event {
	char callerid[256];
	char dest[256];
	char language[256];
	struct sockaddr_in addr;
	char dnid[256];
};

struct addr_event {
	struct sockaddr_in addr;
};

struct authrequest {
	int authmethods;
	char challenge[256];
	char username[256];
};

struct url_event {
	char url[1024];
	int link;
};

struct registration_reply {
	int status;
	char ourip[256];
	char callerid[64];
	short ourport;
	int refresh;
};

struct audio_select {
	struct audio_channel *ac;
	int response;
};

struct auth_reply {
	char password[256];
	char challenge[256];
	int methods;
};

struct new_reply {
	int id;
};

struct registration_request {
	char server[256];
	char peer[256];
	char secret[256];
	int refresh;
};

/* This is remarkably like the IAX events but oh well */
typedef struct pc_event {
	int len;
	int callno;
	int event;
	union {
		struct addr_event addr;
		struct image_event image;
		struct audio_event audio;
		struct dtmf_event dtmf;
		struct dial_event dial;
		struct reject_event reject;
		struct hangup_event hangup;
		struct authrequest authrequest;
		struct registration_reply regreply;
		struct registration_request regrequest;
		struct url_event url;
		struct connect_event connect;
		struct auth_reply authreply;
		struct new_reply newreply;
		struct audio_select asel;
		struct dpreq_event dpreq;
		struct dprep_event dprep;
		struct lagrep_event lag;
		struct text_event text;
	} e;
} pc_event;

#define SOURCE_GUI		0
#define SOURCE_PC		1
#define SOURCE_PRI_OUT	2	/* Priority out */
#define SOURCE_PRI_IN	3	/* Priority in */

extern int pc_write_event(int src, pc_event *e);

static inline int pc_simple_event(int src, int callno, int event)
{
	pc_event e;
	e.len = sizeof(pc_event);
	e.callno = callno;
	e.event = event;
	return pc_write_event(src, &e);
}

static inline int pc_send_unlink(int src, int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_UNLINK);
}

static inline int pc_send_link_reject(int src, int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_UNLINK);
}

static inline int pc_send_lagreq(int src, int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_LAGREQ);
}

static inline int pc_transfer(int src, int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_TRANSFER);
}

static inline int pc_onhook(int src, int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_ONHOOK);
}

static inline int pc_offhook(int src, int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_OFFHOOK);
}


static inline int pc_load_complete(int src, int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_LOADCOMPLETE);
}

static inline int pc_answer(int src, int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_ANSWER);
}


static inline int pc_ring_announce(int src, int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_RINGA);
}

static inline int pc_accept(int src, int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_ACCEPT);
}

static inline int pc_select(int src, int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_SELECT);
}

static inline int pc_conference(int src, int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_CONFERENCE);
}

static inline int pc_unconference(int src, int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_UNCONFERENCE);
}

static inline int pc_dial_tbd(int src, int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_DIAL_TBD);
}

extern int pc_send_linked_url(int src, int callno, char *url);

extern int pc_dial(int src, char digit);

extern int pc_send_text(int src, int callno, char *text);

extern int pc_addr(int src, int callno, struct sockaddr_in addr);

extern int pc_init(void);

extern int pc_write_event(int src, pc_event *e);

extern int pc_read_event(int src, pc_event *e);

extern int pc_send_dtmf(int src, int callno, char digit);

extern int pc_send_dpreq(int src, int callno, char *digits);

extern int pc_hangup(int src, int callno, char *why);

extern int pc_reject(int src, int callno, char *why);

extern int pc_auth_reply(int src, int callno, char *password, char *challenege, int authmethods);

extern int pc_call(int src, int pcid, char *callerid, char *destination, char *lang);

extern int pc_complete(int src, int callno, char *dest);

extern int pc_register(int src, int pcid, char *server, char *peer, char *secret, int to);

extern int pc_session_new(void);

extern int pc_get_fd(void);

extern int pc_select_audio(struct audio_channel *c);

extern void *phonecore_thread(void *unused);

#endif /* _PHONECORE_H */
