/*
 * Gnophone: A client for the Asterisk PBX
 *
 * Copyright (C) 2000, Linux Support Services, Inc.
 *
 * Written by Mark Spencer
 *
 * Linux/UNIX version distributed under the terms of
 * the GNU General Public License
 *
 * audio-toapp.c: Sound to Application Driver by Andre Bierwirth
 *
 */

#include <windows.h>
#include <io.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <winpoop.h>
#include <frame.h>
#include <phonecore.h>
#include "../audio.h"


static int dx_open(struct audio_channel *ac);
static int dx_close(struct audio_channel *ac);
static int dx_activate(struct audio_channel *ac);
static int dx_deactivate(struct audio_channel *ac);
static int dx_configure(struct audio_channel *ac);
static int dx_setspeed(struct audio_channel *ac, int speed);
static int dx_sendaudio(struct audio_channel *ac, int format, void *data, int len);
static int dx_readaudio(struct audio_channel *ac, int format, void *buffer, int *len);

static char dx_driver[] = "Gnophone/toApp";

static struct audio_channel *ac;

struct dx_pvt {
	/* We Callbacks to  */
	void *instance;
	void (*openaudio)(void *instance);
	void (*closeaudio)(void *instance);
	void (*activateaudio)(void *instance);
	void (*deactivateaudio)(void *instance);
	void (*configureaudio)(void *instance);
	void (*setspeedaudio)(void *instance, int speed);
	void (*sendaudio)(void *instance, void *data, int len);
	void (*readaudio)(void *instance, void *buffer, int *len);
};

static struct audio_channel *dx_channel_new(int speed)
{
	struct audio_channel *ac;
	struct dx_pvt *pvt = (struct dx_pvt *)malloc(sizeof(struct dx_pvt));
	ac = audio_new();
	bzero(pvt, sizeof(struct dx_pvt));
	if (ac && pvt) {
		strncpy(ac->name, "Sound To App Driver", sizeof(ac->name));
		snprintf(ac->name, sizeof(ac->name), "SoundToApp version %d", 1);

		ac->priority = 10;
		ac->driver = dx_driver;
		ac->open = dx_open;
		ac->close = dx_close;
		ac->play_digit = std_play_digit;
		ac->activate = dx_activate;
		ac->ring = std_ring;
		ac->busy = std_busy;
		ac->hz = speed;
		ac->fastbusy = std_fastbusy;
		ac->ringing = std_ringing;
		ac->deactivate = dx_deactivate;
		ac->configure = dx_configure;
		ac->sendaudio = dx_sendaudio;
		ac->readaudio = dx_readaudio;
		/* ac->flush = NULL; */
		ac->setspeed = dx_setspeed;
		ac->sformats = AST_FORMAT_SLINEAR;
		/* ac->cananswer = 0; */
		ac->duplex = 1;			/* ESD is always full duplex */
		/* ac->simduplex = NULL; */
		ac->fd = -1;
		ac->pvt = pvt;
		ac->echocancelled = 0;

	} else if (!pvt) { free(ac); ac = NULL; }
	return ac;
}

static int dx_open(struct audio_channel *ac)
{
	struct dx_pvt *pvt = ac->pvt;
	if (pvt->openaudio)
		pvt->openaudio(pvt->instance);
	return 0;
}

static int dx_close(struct audio_channel *ac)
{
	struct dx_pvt *pvt = ac->pvt;
	if (pvt->closeaudio)
		pvt->closeaudio(pvt->instance);
	return 0;
}

static int dx_activate(struct audio_channel *ac)
{
	/* Nothing really necessary to activate us */
	struct dx_pvt *pvt = ac->pvt;

	if (pvt->activateaudio)
		pvt->activateaudio(pvt->instance);
	return 0;
}

static int dx_deactivate(struct audio_channel *ac)
{
	/* Nothing really necessary to deactivate us either */
	struct dx_pvt *pvt = ac->pvt;

	if (pvt->deactivateaudio)
		pvt->deactivateaudio(pvt->instance);
	return 0;
}

static int dx_configure(struct audio_channel *ac)
{
	/* XXX Bug: Need a configuration window, mixer, etc.. XXX */
	struct dx_pvt *pvt = ac->pvt;

	if (pvt->configureaudio)
		pvt->configureaudio(pvt->instance);
	return 0;
}

static int dx_setspeed(struct audio_channel *ac, int speed)
{
	struct dx_pvt *pvt = ac->pvt;

	if (ac->hz != speed) {
		ac->hz = speed;
	    if (pvt->setspeedaudio)
			pvt->setspeedaudio(pvt->instance, speed);

    }
	return 0;
}

static int dx_readaudio(struct audio_channel *ac, int format, void *data, int *len)
{
	struct dx_pvt *pvt = ac->pvt;

	if (pvt->readaudio)
		pvt->readaudio(pvt->instance, data,len);

	return 0;

}

static int dx_sendaudio(struct audio_channel *ac, int format, void *data, int len)
{
	struct dx_pvt *pvt = ac->pvt;

	if (pvt->sendaudio)
		pvt->sendaudio(pvt->instance, data,len);

	return 0;
}

char *name()
{
	return "toapp";
}

int sound_toapp_init(void *instance,
					 void (*openaudio)(void *instance),
					 void (*closeaudio)(void *instance),
					 void (*activateaudio)(void *instance),
					 void (*deactivateaudio)(void *instance),
					 void (*configureaudio)(void *instance),
					 void (*setspeedaudio)(void *instance, int speed),
					 void (*sendaudio)(void *instance, void *data, int len),
					 void (*readaudio)(void *instance, void *buffer, int *len))
{
	struct dx_pvt *pvt;
	ac = dx_channel_new(AUDIO_DEFAULT_SPEED);
	if (ac) {
		pvt = ac->pvt;
		pvt->instance  = instance;
		pvt->openaudio = openaudio;
		pvt->closeaudio = closeaudio;
		pvt->activateaudio = activateaudio;
		pvt->deactivateaudio = deactivateaudio;
		pvt->configureaudio = configureaudio;
		pvt->setspeedaudio = setspeedaudio;
		pvt->sendaudio = sendaudio;
		pvt->readaudio = readaudio;
		audio_register_channel(ac);
		pc_select_audio(ac);
	}
	return 0;
}

int sound_toapp_send(void *data, int len)
{
	deliver_sound(AST_FORMAT_SLINEAR, data, len);
	return 0;
}

