/*
 * Gnophone: A client for the Asterisk PBX
 *
 * Copyright (C) 2000, Linux Support Services, Inc.
 *
 * Written by Mark Spencer
 *
 * Linux/UNIX version distributed under the terms of
 * the GNU General Public License
 *
 * phonecore.c: Core telephony thread
 *
 */

#ifdef WIN32

#include <iax-client.h>
#include <windows.h>
#include <winsock2.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <gsm.h>
#include <string.h>
#include <errno.h>

#include <winpoop.h>
#include "astio.h"

#else
#include <iax/iax-client.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <gsm.h>
#include <sys/socket.h>
#include <netinet/ip.h>
#include <string.h>
#include <errno.h>
#include "io.h"

#endif

#include "phonecore.h"
#include "asched.h"
#include "audio.h"
#include "../sounds/zero.h"
#include "../sounds/one.h"
#include "../sounds/two.h"
#include "../sounds/three.h"
#include "../sounds/four.h"
#include "../sounds/five.h"
#include "../sounds/six.h"
#include "../sounds/seven.h"
#include "../sounds/eight.h"
#include "../sounds/nine.h"
#include "../sounds/star.h"
#include "../sounds/pound.h"

#include "../sounds/busy.h"
#include "../sounds/ringt.h"
#include "../sounds/ring.h"

#define CHUNK_MS 20
#define CHUNKLEN 160

/* Keep three levels of conference buffer */
#define CNF_BUF 4

#define SEND(a) pc_send_sound(c, a, sizeof(a)/2, 0 )
#define SEND2(a) pc_send_sound(c, a, sizeof(a)/2, 1 )

static int switchtime = DEFAULT_SWITCHTIME;

static struct io_context *io;
static struct sched_context *sched;

/* Sched ID of timeout for audio */
static int *id;

/* Desired read format */
static int format = AST_FORMAT_SLINEAR;
/* Max chunk to ask the driver to read at once.  Deliver
   only exactly this much data to in a frame passed
   to the top level. */
static int chunk = 320;

/* This is whether we must have a full chunk or not */
static int enforcechunk = 1;

static int ready = 30;
static CRITICAL_SECTION ready_lock;

/* Kinda handy */
void *unused;

#ifdef WIN32

static HANDLE sockthread = NULL;
static void (*debughandler)(char *text) = NULL;

/* This is a little strange, but to debug you call DEBU(G "Hello World!\n"); */ \
#define G __FILE__, __LINE__,

#define DEBU __debug
int __debug(char *file, int lineno, char *fmt, ...)
{
	char text[4098];
	va_list args;
	va_start(args, fmt);
		fprintf(stderr, "%s line %d: ", file, lineno);
		if (debughandler) {
			sprintf(text, "%s line %d: ", file, lineno);
        	debughandler(text);
		}
		vfprintf(stderr, fmt, args);
		if (debughandler) {
			vsprintf(text, fmt, args);
        	debughandler(text);
		}
	va_end(args);
	return 0;
}

int pc_debug_callback(void (*handler)(char *text)) {
	debughandler = handler;
	iax_debug_callback(handler);
	return 0;
}

void *socks_thread(void *unused) {
//	int *id, int fd, short +s, void *cbdata
	for (;;) {
		handle_socks(0,0,0,NULL);
	}
	return NULL;
}

#endif


#define CNF_INC(z) do { \
	z = (z + 1) % CNF_BUF; \
} while(0)

static int cnfcount = 0;

struct peer {
	gsm gsmin;
	gsm gsmout;
	int mute;
	struct iax_session *session;
	int cnfin;
	int cnfout;
	int inconf;
	short cnfbuf[CNF_BUF * CHUNKLEN];
} *peers[PC_MAX_CALLS];

static int current;

#ifdef WIN32
static struct simsock *socks[4];
#else
static int socks[4];
#endif


struct peer *new_peer(struct iax_session *session)
{
	struct peer *p;
	p = malloc(sizeof(struct peer));
	if (p) {
		bzero(p, sizeof(struct peer));
		p->session = session ? session : iax_session_new();
		p->mute = 1;
		p->cnfin = 1;
		p->cnfout = 0;
		if (!p->session) {
			free(p);
			p = NULL;
		}
	}
	return p;
};

int pc_read_event(int src, pc_event *e)
{
	int res;
#ifdef WIN32
	res = sim_sock_read(socks[src], e, sizeof(pc_event));
#else
	res = read(socks[src], e, sizeof(pc_event));
#endif
	if (res != sizeof(pc_event)) {
		DEBU(G "Only read %d of %d bytes on %d: %s\n", res, sizeof(pc_event), src, strerror(errno));
		return -1;
	}
	if ((e->callno > PC_MAX_CALLS) || (e->callno < -1)) {
		DEBU(G "!!! Call number %d invalid (event %d len %d)!!!\n", e->callno, e->event, e->len);
		return -1;
	}
	return 0;
}

int pc_write_event(int src, pc_event *e)
{
	int res;
#if 0
	printf("Writing even %d on %d\n", e->event, src);
#endif

#ifdef WIN32
	res = sim_sock_write(socks[src], e, sizeof(pc_event));
#else
	res = write(socks[src], e, sizeof(pc_event));
#endif
	if (res != sizeof(pc_event)) {
		DEBU(G "Only wrote %d of %d bytes: %s\n", res, sizeof(*e), strerror(errno));
		exit(1);
		return -1;
	}
	return 0;
}


#define INIT_PE(ev) do { \
	pe.len = sizeof(pe); \
	pe.callno = get_callno(e->session); \
	pe.event = ev; \
} while (0)

static int audio_send(struct audio_channel *c, int format, void *data, int datalen, int loud, int stop);
static int get_level(short *data, int samples);

static inline int get_callno(struct iax_session *s)
{
	int x;
	for (x=0;x<PC_MAX_CALLS;x++)
		if (peers[x] && (peers[x]->session == s))
			return x;
	return -1;
}

static void handle_voice(struct iax_event *e)
{
	int id;
	short fr[160];
	int len;
	pc_event pe;
	struct peer *p;
	id = get_callno(e->session);
	if (id > -1) {
		p = peers[id];
		if (!p->gsmin)
			p->gsmin = gsm_create();

		len = 0;
		if (e->event.voice.datalen % 33) {
			DEBU(G "Error: audio not a multiple of 33 bytes\n");
			return;
		}
		while(len < e->event.voice.datalen) {
			gsm_decode((gsm) p->gsmin,(gsm_byte *) e->event.voice.data + len,(gsm_signal *) fr);
			if ((id == current))
				audio_send(audioc, AST_FORMAT_SLINEAR, fr, sizeof(fr), 0, 1);
			memcpy(p->cnfbuf + p->cnfin * CHUNKLEN, fr, sizeof(fr));
			CNF_INC(p->cnfin);
			/* If we've overrun our buffer, increment the output conference since we've overwritten
			   its (now outdated) data anyway */
			if (p->cnfin == p->cnfout) {
#if 0
				printf("Out of storage space, forcing inc\n");
#endif
				CNF_INC(p->cnfout);
			}
			EnterCriticalSection(&ready_lock);
			if (ready && !len) {
				pe.len = sizeof(pe);
				pe.callno = id;
				pe.event = PC_EVENT_AUDIO;
				pe.e.audio.level = get_level(fr, 160);
				pc_write_event(SOURCE_PC, &pe);
				ready--;
			}
			LeaveCriticalSection(&ready_lock);
			len += 33;
		}
	} else
		DEBU(G "Voice on non-existant session %p\n", e->session);
}

static void free_peer(int id)
{
	if (peers[id]) {
		if (peers[id]->inconf)
			cnfcount--;
		if (peers[id]->gsmin)
			gsm_destroy(peers[id]->gsmin);
		if (peers[id]->gsmout)
			gsm_destroy(peers[id]->gsmout);
		free(peers[id]);
		peers[id] = NULL;
	}
}

static int new_peer_id(struct iax_session *session);

static int handle_event(struct iax_event *e)
{
	pc_event pe;
	int cn;
	switch(e->etype) {
	case IAX_EVENT_CONNECT:
		pe.len = sizeof(pe);
		pe.event = PC_EVENT_CONNECT;
		pe.callno = new_peer_id(e->session);
		if (e->event.connect.callerid)
			strncpy(pe.e.connect.callerid, e->event.connect.callerid,
					sizeof(pe.e.connect.callerid)-1);
		else
			strcpy(pe.e.connect.callerid, "");

		if (e->event.connect.dnid)
			strncpy(pe.e.connect.dnid, e->event.connect.dnid,
					sizeof(pe.e.connect.dnid)-1);
		else
			strcpy(pe.e.connect.dnid, "");

		pe.e.connect.addr = iax_get_peer_addr(e->session);
		if (pc_write_event(SOURCE_PC, &pe))
			DEBU(G "Unable to deliver connect event\n");
		return 0;
	case IAX_EVENT_HANGUP:
		cn = get_callno(e->session);
		debughandler("Phonecore: Send Hangup");
		free_peer(cn);
		return pc_hangup(SOURCE_PC, cn, e->event.hangup.byemsg);
	case IAX_EVENT_REJECT:
		cn = get_callno(e->session);
		free_peer(cn);
		return pc_reject(SOURCE_PC, cn, e->event.reject.reason);
	case IAX_EVENT_ACCEPT:
		return pc_accept(SOURCE_PC,get_callno(e->session));
	case IAX_EVENT_ANSWER:
		peers[get_callno(e->session)]->mute = 0;
		return pc_answer(SOURCE_PC, get_callno(e->session));
	case IAX_EVENT_AUTHRQ:
		INIT_PE(PC_EVENT_AUTHRQ);
		pe.e.authrequest.authmethods = e->event.authrequest.authmethods;
		if (e->event.authrequest.challenge)
			strncpy(pe.e.authrequest.challenge, e->event.authrequest.challenge,
					sizeof(pe.e.authrequest.challenge)-1);
		else
			strcpy(pe.e.authrequest.challenge, "");
		if (e->event.authrequest.username)
			strncpy(pe.e.authrequest.username, e->event.authrequest.username,
					sizeof(pe.e.authrequest.username)-1);
		else
			strcpy(pe.e.authrequest.username, "");
		if (pc_write_event(SOURCE_PC, &pe))
			DEBU(G "Unable to report auth request\n");
		return 0;
	case IAX_EVENT_IMAGE:
		debughandler("PhoneCore: GetImage");
		INIT_PE(PC_EVENT_IMAGE);
		if (e->event.image.datalen > sizeof(pe.e.image.data)) {
			DEBU(G "!! Image too large to copy (%d bytes) !!\n",
				e->event.image.datalen);
			return 0;
		}
		memcpy(pe.e.image.data, e->event.image.data, e->event.image.datalen);
		pe.e.image.datalen = e->event.image.datalen;
		if (pc_write_event(SOURCE_PC, &pe))
			DEBU(G "Unable to return image\n");
		return 0;
	case IAX_EVENT_LAGRP:
		INIT_PE(PC_EVENT_LAGREP);

		pe.e.lag.lag = e->event.lag.lag;
		pe.e.lag.jitter = e->event.lag.jitter;

		if (pc_write_event(SOURCE_PC, &pe))
			DEBU(G "Unable to send LAGREP\n");

		break;

	case IAX_EVENT_UNLINK:
		pc_send_unlink(SOURCE_PC, get_callno(e->session));

		break;

	case IAX_EVENT_LINKREJECT:
		pc_send_link_reject(SOURCE_PC, get_callno(e->session));

		break;

	case IAX_EVENT_TEXT:
		INIT_PE(PC_EVENT_TEXT);
		if (e->event.text.text)
			strncpy(pe.e.text.text, e->event.text.text,
					sizeof(pe.e.text.text)-1);

		if (pc_write_event(SOURCE_PC, &pe))
				DEBU(G "Unable to return text\n");

		return 0;

	case IAX_EVENT_URL:
		INIT_PE(PC_EVENT_URL);
		if (e->event.url.url)
			strncpy(pe.e.url.url, e->event.url.url,
					sizeof(pe.e.url.url)-1);
		else
			strcpy(pe.e.url.url, "");

		pe.e.url.link = e->event.url.link;

		if (pc_write_event(SOURCE_PC, &pe))
			DEBU(G "Unable to return URL\n");
		return 0;
	case IAX_EVENT_RINGA:
		return pc_ring_announce(SOURCE_PC,get_callno(e->session));
	case IAX_EVENT_REGREP:
		INIT_PE(PC_EVENT_REGREP);
		free_peer(pe.callno);
		if (e->event.regreply.ourip)
			strncpy(pe.e.regreply.ourip, e->event.regreply.ourip,
					sizeof(pe.e.regreply.ourip)-1);
		else
			strcpy(pe.e.regreply.ourip, "");

		pe.e.regreply.ourport = e->event.regreply.ourport;
		pe.e.regreply.refresh = e->event.regreply.refresh;
		pe.e.regreply.status = e->event.regreply.status;

		if (e->event.regreply.callerid)
			strncpy(pe.e.regreply.callerid, e->event.regreply.callerid, sizeof(pe.e.regreply.callerid)-1);
		else
			strcpy(pe.e.regreply.callerid, "");

		if (pc_write_event(SOURCE_PC, &pe))
			DEBU(G "Unable to report registration reply\n");
		return 0;
	case IAX_EVENT_VOICE:
		handle_voice(e);
		return 0;
	case IAX_EVENT_TRANSFER:
		printf("Transferred to %s:%d\n", e->event.transfer.newip, e->event.transfer.newport);
		pc_transfer(SOURCE_PC, get_callno(e->session));
		return 0;
	case IAX_EVENT_DPREP:

		if (!e->event.dprep.canexist)
		{

			INIT_PE(PC_EVENT_DPREP);

			pe.callno = -1;
			pe.e.dprep.exists = e->event.dprep.exists;
			pe.e.dprep.nonexistant = e->event.dprep.nonexistant;
			pe.e.dprep.canexist = e->event.dprep.canexist;
			pe.e.dprep.ignorepat = e->event.dprep.ignorepat;

			strcpy(pe.e.dprep.number, e->event.dprep.number);

			if (pc_write_event(SOURCE_PC, &pe))
				DEBU(G "Unable to report dialplan reply.\n");
		}

		return 0;
	default:
		DEBU(G "Don't know what to do with IAX event %d\n", e->etype);
	}
	return -1;
}


static int handle_iax(int *id, int fd, short events, void *cbdata)
{
	struct iax_event *e;
	while((e = iax_get_event(0))) {
		handle_event(e);
		iax_event_free(e);
	}
	return 1;
}

static int pc_audio_select(struct audio_channel *c);

static int pc_send_sound_cb(void *data)
{
	struct audio_channel *c = data;
	int ms;
	struct timeval tv;
	int lentosend;

again:
	lentosend = c->totallen - c->lensofar;
	if (lentosend > CHUNKLEN)
		lentosend = CHUNKLEN;
	audio_send(c, AST_FORMAT_SLINEAR, c->sound + c->lensofar, lentosend * 2, 1, 0);
	c->lensofar += lentosend;
	if (c->lensofar == c->totallen) {
		if (c->repeat) {
			c->lensofar = 0;
		} else {
			c->id = -1;
#if 0
			if (c->flush)
				c->flush(c);
#endif
#if 0
			/* Put it in reading mode */
			if (!c->duplex && c->simduplex)
				c->simduplex(c, 0);
#endif
			return 0;
		}
	}

	gettimeofday(&tv, NULL);
	/* Here's how far along we really are... */
	ms = (tv.tv_sec - c->st.tv_sec) * 1000 + (tv.tv_usec - c->st.tv_usec)/1000;

	/* Our position increases by samples/8 ms */
	c->pos += lentosend / 8;
	if (ms >= c->pos) {
#if 1
		DEBU(G "We must really be slow...\n");
#endif
		/* We already have to send another frame */
		goto again;
	}
	c->id = ast_sched_add(sched, c->pos - ms, pc_send_sound_cb, c);
//	c->id = ast_sched_add(sched, 2, pc_send_sound_cb, c);
	return 0;
}

static int pc_send_sound(struct audio_channel *c, short *a, int len, int repeat)
{
	/* Put it in writing mode immediately */
	if (c->id > -1)
		ast_sched_del(sched, c->id);
	if (c->flush)
		c->flush(c);
	c->totallen = len;
	c->lensofar = 0;
	c->sound = a;
	c->repeat = repeat;
	c->pos = 0;
	gettimeofday(&c->st, NULL);
	/* Send the first couple of packets immediately to get things going */
	pc_send_sound_cb(c);
#if 1 // Try only
	if (c->id > -1) {
		c->pos = 0;
		ast_sched_del(sched, c->id);
		pc_send_sound_cb(c);
	}
	if (c->id > -1) {
		c->pos = 0;
		ast_sched_del(sched, c->id);
		pc_send_sound_cb(c);
	}
#endif
	return 0;
}

int std_ring(struct audio_channel *c)
{
	SEND2(ring);
	return 0;
}

int std_ringing(struct audio_channel *c)
{
	SEND2(ringt);
	return 0;
}


int std_busy(struct audio_channel *c)
{
	SEND2(busy);
	return 0;
}

int std_fastbusy(struct audio_channel *c)
{
	SEND2(busy);
	return 0;
}

int std_play_digit(struct audio_channel *c, char digit)
{
	/* Nothing to do if they don't have audio */
	if (!c->sendaudio)
		return 0;
	switch(digit) {
	case '0':
		SEND(zero);
		break;
	case '1':
		SEND(one);
		break;
	case '2':
		SEND(two);
		break;
	case '3':
		SEND(three);
		break;
	case '4':
		SEND(four);
		break;
	case '5':
		SEND(five);
		break;
	case '6':
		SEND(six);
		break;
	case '7':
		SEND(seven);
		break;
	case '8':
		SEND(eight);
		break;
	case '9':
		SEND(nine);
		break;
	case '*':
		SEND(star);
		break;
	case '#':
		SEND(pound);
		break;
	default:
		DEBU(G "Unknown digit: %c\n", digit);
	}
	return 0;
}

static int pc_send_digit(struct audio_channel *c, char digit)
{
	if (audioc)
		return audioc->play_digit(audioc, digit);
	return 0;
}


int pc_simple_event(int src,int callno,int event)
{
	pc_event e;
	e.len = sizeof(pc_event);
	e.callno = callno;
	e.event = event;
	return pc_write_event(src, &e);
}

int pc_onhook(int src,int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_ONHOOK);
}

int pc_offhook(int src,int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_OFFHOOK);
}

int pc_send_unlink(int src,int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_UNLINK);
}

int pc_send_link_reject(int src,int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_UNLINK);
}

int pc_send_lagreq(int src,int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_LAGREQ);
}

int pc_transfer(int src,int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_TRANSFER);
}

int pc_load_complete(int src,int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_LOADCOMPLETE);
}

int pc_answer(int src,int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_ANSWER);
}

int pc_ring_announce(int src,int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_RINGA);
}

int pc_accept(int src,int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_ACCEPT);
}

int pc_select(int src,int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_SELECT);
}

int pc_conference(int src,int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_CONFERENCE);
}

int pc_unconference(int src,int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_UNCONFERENCE);
}

int pc_dial_tbd(int src,int callno)
{
	return pc_simple_event(src, callno, PC_EVENT_DIAL_TBD);
}

int pc_send_dpreq(int src, int callno, char *digits)
{
	pc_event e;
	e.len = sizeof(pc_event);
	e.callno = callno;
	e.event = PC_EVENT_DPREQ;
	snprintf(e.e.dpreq.dest, sizeof(e.e.dpreq.dest), "%s", digits);
	return pc_write_event(src, &e);
}

int pc_send_linked_url(int src, int callno, char *url)
{
	pc_event e;
	e.len = sizeof(pc_event);
	e.callno = callno;
	e.event = PC_EVENT_URL;
	e.e.url.link = 1;
	snprintf(e.e.url.url, sizeof(e.e.url.url), "%s", url);
	return pc_write_event(src, &e);
}

static int pc_audio_ring(struct audio_channel *c)
{
	if (!c)
		return 0;
	if (c->ring)
		return c->ring(c);
	return 0;
}

static int pc_audio_ringing(struct audio_channel *c)
{
	if (!c)
		return 0;
	if (c->ringing)
		return c->ringing(c);
	return 0;
}

static int pc_audio_shutup(struct audio_channel *c)
{
	if (!c)
		return 0;
	if (c->id > -1)
		ast_sched_del(sched, c->id);
	c->id = -1;
	if (c->flush)
		c->flush(c);
#if 0
	/* Put it in reading mode */
	if (!c->duplex && c->simduplex)
		c->simduplex(c, 0);
#endif
	return 0;
}

static int pc_audio_deactivate(struct audio_channel *c)
{
	if (!c)
		return 0;
	if (c->id > -1)
		ast_sched_del(sched, c->id);
	c->id = -1;
	if (c->deactivate)
		c->deactivate(c);
	return 0;
}

static int new_peer_id(struct iax_session *session)
{
	int x;
	for (x=0;x<PC_MAX_CALLS;x++)
		if (!peers[x])
			break;
	if (x < PC_MAX_CALLS) {
		peers[x] = new_peer(session);
		if (peers[x])
			return x;
	}
	return -1;
}

int handle_socks(int *id, int fd, short events, void *cbdata)
{
	pc_event e;
	int res;
	res = pc_read_event(SOURCE_PC, &e);
	if (res < 0)
		return 1;
	switch(e.event) {
	case PC_EVENT_SELAUDIO:
		e.event = PC_EVENT_AUDIO_REPLY;
		e.e.asel.response = pc_audio_select(e.e.asel.ac);
		if (pc_write_event(SOURCE_PRI_IN, &e))
			DEBU(G "Unable to send audio result\n");
		break;
	case PC_EVENT_NEW:
		e.event = PC_EVENT_NEW_REPLY;
		e.e.newreply.id = new_peer_id(NULL);
		if (pc_write_event(SOURCE_PRI_IN, &e))
			DEBU(G "Unable to send new reply\n");
		break;
	case PC_EVENT_AUDIO_READY:
		/* Send another audio update whenever */
		EnterCriticalSection(&ready_lock);
		ready++;
		LeaveCriticalSection(&ready_lock);
		break;
	case PC_EVENT_REGREQ:
#if 0
		DEBU(G "Session: %p, server: %s, peer: %s, secret: %s, to: %d\n",
			peers[e.callno]->session, e.e.regrequest.server, e.e.regrequest.peer,
			e.e.regrequest.secret, e.e.regrequest.refresh);
#endif
		if (iax_register(peers[e.callno]->session, e.e.regrequest.server, e.e.regrequest.peer, e.e.regrequest.secret, e.e.regrequest.refresh)) {
			DEBU(G "IAX Register failed: %s\n", strerror(errno));
		}
		break;
	case PC_EVENT_CONNECT:
		if (iax_call(peers[e.callno]->session, e.e.connect.callerid, e.e.connect.dest, e.e.connect.language, 0)) {
			DEBU(G "IAX Call failed: (session: %p, callerid: %s, dest: %s, language: %s): %s\n", peers[e.callno]->session, e.e.connect.callerid, e.e.connect.dest, e.e.connect.language, strerror(errno));
		}
		pc_addr(SOURCE_PC, e.callno, iax_get_peer_addr(peers[e.callno]->session));
		break;

	case PC_EVENT_COMPLETE:
		DEBU(G "DEBUG: e.callno = %d\n", e.callno);
		DEBU(G "DEBUG: session  = %p\n", peers[e.callno]->session);
		DEBU(G "DEBUG: dest     = %s\n", e.e.connect.dest);
		if (iax_dial(peers[e.callno]->session, e.e.connect.dest)) {
			DEBU(G "IAX Dial failed: %s\n", strerror(errno));
		}
		pc_addr(SOURCE_PC, e.callno, iax_get_peer_addr(peers[e.callno]->session));
		break;

	case PC_EVENT_HANGUP:
		debughandler("Phonecore: Sent PC_EVENT_HANGUP");
		if (iax_hangup(peers[e.callno]->session, e.e.hangup.why))
			DEBU(G "IAX hangup failed: %s\n", strerror(errno));
		free_peer(e.callno);
		break;
	case PC_EVENT_AUDIO_DIGIT:
		pc_send_digit(audioc, e.e.dtmf.dtmf);
		break;
	case PC_EVENT_AUDIO_RING:
		pc_audio_ring(audioc);
		break;
	case PC_EVENT_AUDIO_RINGING:
		pc_audio_ringing(audioc);
		break;
	case PC_EVENT_AUDIO_SHUTUP:
		pc_audio_shutup(audioc);
		break;
	case PC_EVENT_AUDIO_DEACTIVATE:
		pc_audio_deactivate(audioc);
		break;
	case PC_EVENT_SELECT:
		current = e.callno;
		break;
	case PC_EVENT_DTMF:
		if (iax_send_dtmf(peers[e.callno]->session, e.e.dtmf.dtmf))
			DEBU(G "IAX dtmf failed: %s\n", strerror(errno));
		break;
	case PC_EVENT_RINGA:
		if (iax_ring_announce(peers[e.callno]->session))
			DEBU(G "IAX ring announce failed: %s\n", strerror(errno));
		break;
	case PC_EVENT_ACCEPT:
		printf("Accepting call on session %p\n", peers[e.callno]->session);
		if (iax_accept(peers[e.callno]->session))
			DEBU(G "IAX accept failed: %s\n", strerror(errno));
		break;
	case PC_EVENT_ANSWER:
		peers[e.callno]->mute = 0;
		if (iax_answer(peers[e.callno]->session))
			DEBU(G "IAX answer failed: %s\n", strerror(errno));
		break;
	case PC_EVENT_LOADCOMPLETE:
		if (iax_load_complete(peers[e.callno]->session))
			DEBU(G "IAX load complete failed: %s\n", strerror(errno));
		break;
	case PC_EVENT_AUTHRP:
		if (iax_auth_reply(peers[e.callno]->session, e.e.authreply.password,
				e.e.authreply.challenge, e.e.authreply.methods))
			DEBU(G "IAX hangup failed: %s\n", strerror(errno));
		break;
	case PC_EVENT_CONFERENCE:
		if (!peers[e.callno]->inconf) {
			peers[e.callno]->inconf=1;
			/* If it's our current talker, put us in the conference now too */
			if (e.callno == current)
				current = -1;
			cnfcount++;
		}
		break;
	case PC_EVENT_UNCONFERENCE:
		if (peers[e.callno]->inconf) {
			peers[e.callno]->inconf=0;
			cnfcount--;
		}
		break;
	case PC_EVENT_TEXT:
		printf("I would be sending the following text: %s to pcid(%d)\n", e.e.text.text, e.callno);

		if ((e.callno >= 0) && peers[e.callno])
			if (iax_send_text(peers[e.callno]->session, e.e.text.text))
				DEBU(G "IAX failed to send text: %s\n", strerror(errno));

		break;
	case PC_EVENT_IMAGE:
		debughandler("Phonecore: Send Image");
		if ((e.callno >= 0) && peers[e.callno])
			if (iax_send_image(peers[e.callno]->session, AST_FORMAT_JPEG, e.e.image.data, e.e.image.datalen))
				DEBU(G "IAX failed to send image: %s\n", strerror(errno));

		break;

	case PC_EVENT_URL:
		if ((e.callno >= 0) && peers[e.callno])
			if (iax_send_url(peers[e.callno]->session, e.e.url.url, e.e.url.link))
				DEBU(G "IAX failed to send url: %s\n", strerror(errno));

		break;

	case PC_EVENT_DPREQ:

		if (iax_dialplan_request(peers[e.callno]->session, e.e.dpreq.dest))
			DEBU(G "IAX failed to send a dialplan request: %s\n", strerror(errno));

		break;

	case PC_EVENT_LAGREQ:

		/* Make sure that we have a valid session before sending the request */
		if ((e.callno >= 0) && peers[e.callno])
			if (iax_lag_request(peers[e.callno]->session))
				DEBU(G "IAX failed to send a lag request: %s\n", strerror(errno));

		break;

	case PC_EVENT_UNLINK:

		if ((e.callno >= 0) && peers[e.callno])
			if (iax_send_unlink(peers[e.callno]->session))
				DEBU(G "IAX failed to send an unlink request: %s\n", strerror(errno));
		break;
	case PC_EVENT_LINKREJECT:

		if ((e.callno >= 0) && peers[e.callno])
			if (iax_send_link_reject(peers[e.callno]->session))
				DEBU(G "IAX failed to send a link reject request: %s\n", strerror(errno));
		break;

	default:
		DEBU(G "Don't know what to do with sockie event %d\n", e.event);
	}
	return 1;
}

int pc_init(void)
{
	int port;
	int fd;
	if ((port = iax_init(0)) < 0) {
		DEBU(G "Failed to initialize IAX subsystem\n");
		return -1;
	}
	fd = iax_get_fd();

	DEBU(G "Listening on port %d\n", port);

#ifndef WIN32
	int tos = IPTOS_LOWDELAY;
	if (setsockopt(fd, SOL_IP, IP_TOS, &tos, sizeof(tos)))
		DEBU(G "Warning: Unable to set IP TOS bits\n");
#else
//	BOOL tnd = TRUE;
//	if (setsockopt(fd, IPPROTO_TCP, TCP_NODELAY, &tnd, sizeof(tnd)))
//		DEBU(G "Warning: Unable to set TCP_NODELAY\n");
#endif

	if (socketpair(AF_UNIX, SOCK_DGRAM, 0, socks)) {
		DEBU(G "Failed to create socket pair\n");
		return -1;
	}
	DEBU(G "First Socket %d", socks[0]);
	DEBU(G "Second Socket %d", socks[1]);

	if (socketpair(AF_UNIX, SOCK_DGRAM, 0, socks + 2)) {
		DEBU(G "Failed to create priority socket pair\n");
		return -1;
	}
	DEBU(G "First PriSocket %d", socks[2]);
	DEBU(G "Second PriSocket %d", socks[3]);

	iax_set_formats(AST_FORMAT_GSM);
	io = io_context_create();
	sched = sched_context_create();
	DEBU(G "Initialized phone core\n");

	DEBU(G "IAX FD %d", iax_get_fd());
	ast_io_add(io, iax_get_fd(), handle_iax, AST_IO_IN, NULL);
#ifdef WIN32
	unsigned long pid;
	socks[SOURCE_GUI]->block = TRUE;
	socks[SOURCE_PC]->block = FALSE;
	socks[SOURCE_PRI_OUT]->block = TRUE;
	sockthread=CreateThread(0,0, socks_thread, NULL, pid, &pid);
	SetThreadPriority(sockthread,THREAD_PRIORITY_BELOW_NORMAL);
	InitializeCriticalSection(&ready_lock);
#else
	ast_io_add(io, socks[SOURCE_PC], handle_socks, AST_IO_IN, NULL);
#endif
	return port;
}

int pc_send_dtmf(int src, int callno, char digit)
{
	pc_event e;
	e.len = sizeof(pc_event);
	e.callno = callno;
	e.event = PC_EVENT_DTMF;
	e.e.dtmf.dtmf = digit;
	return pc_write_event(src, &e);
}

int pc_hangup(int src, int callno, char *why)
{
	pc_event e;
	e.len = sizeof(pc_event);
	e.callno = callno;
	e.event = PC_EVENT_HANGUP;
	if (why)
		strncpy(e.e.hangup.why, why, sizeof(e.e.hangup.why)-1);
	else
		strcpy(e.e.hangup.why, "");
	return pc_write_event(src, &e);
}

int pc_reject(int src, int callno, char *why)
{
	pc_event e;
	e.len = sizeof(pc_event);
	e.callno = callno;
	e.event = PC_EVENT_REJECT;
	if (why)
		strncpy(e.e.reject.why, why, sizeof(e.e.reject.why)-1);
	else
		strcpy(e.e.reject.why, "");
	return pc_write_event(src, &e);
}

int pc_dial(int src, char digit)
{
	pc_event e;

	e.len = sizeof(pc_event);
	e.callno = -1;
	e.event = PC_EVENT_DIAL;
	e.e.dial.digit = digit;

	return pc_write_event(src, &e);
}

int pc_send_text(int src, int callno, char *text)
{
	pc_event e;
	e.len = sizeof(pc_event);
	e.callno = callno;
	e.event = PC_EVENT_TEXT;
	snprintf(e.e.text.text, sizeof(e.e.text.text), "%s", text);
	return pc_write_event(src, &e);
}

int pc_addr(int src, int callno, struct sockaddr_in sin)
{
	pc_event e;
	e.len = sizeof(pc_event);
	e.callno = callno;
	e.event = PC_EVENT_ADDR;
	e.e.addr.addr = sin;
	return pc_write_event(src, &e);
}

int pc_auth_reply(int src, int callno, char *passwd, char *challenge, int authmethods)
{
	pc_event e;
	e.len = sizeof(pc_event);
	e.callno = callno;
	e.event = PC_EVENT_AUTHRP;
	strncpy(e.e.authreply.password, passwd, sizeof(e.e.authreply.password)-1);
	if (challenge)
		strncpy(e.e.authreply.challenge, challenge,sizeof(e.e.authreply.challenge)-1);
	else
		strcpy(e.e.authreply.challenge, "");
	e.e.authreply.methods = authmethods;
	return pc_write_event(src, &e);
}

int pc_call(int src, int callno, char *callerid, char *dest, char *language)
{
	DEBU(G "Execute pc_call(%s,%s,%s)\n", callerid, dest, language);

	pc_event e;
	e.len = sizeof(pc_event);
	e.callno = callno;
	e.event = PC_EVENT_CONNECT;
	if (callerid)
		strncpy(e.e.connect.callerid, callerid, sizeof(e.e.connect.callerid)-1);
	else
		strcpy(e.e.connect.callerid, "");
	if (dest)
		strncpy(e.e.connect.dest, dest, sizeof(e.e.connect.dest)-1);
	else
		strcpy(e.e.connect.dest, "");
	if (language)
		strncpy(e.e.connect.language, language, sizeof(e.e.connect.language)-1);
	else
		strcpy(e.e.connect.language, "");
	return pc_write_event(src, &e);
}

int pc_complete(int src, int callno, char *dest)
{
	pc_event e;
	e.len = sizeof(pc_event);
	e.callno = callno;
	e.event = PC_EVENT_COMPLETE;

	if (dest)
		strncpy(e.e.connect.dest, dest, sizeof(e.e.connect.dest)-1);
	else
		strcpy(e.e.connect.dest, "");

	return pc_write_event(src, &e);
}

int pc_select_audio(struct audio_channel *ac)
{
	pc_event e;
	e.event = PC_EVENT_SELAUDIO;
	e.len = sizeof(pc_event);
	e.callno = 0;
	e.e.asel.ac = ac;
	pc_write_event(SOURCE_GUI, &e);
	if (pc_read_event(SOURCE_PRI_OUT, &e))
		return -1;
	if (e.event != PC_EVENT_AUDIO_REPLY) {
		DEBU(G "Huh?  Got '%d' instead of audio reply\n", e.event);
		return -1;
	}
	audioc = e.e.asel.ac;
	return e.e.asel.response;
}

int pc_register(int src, int callno, char *server, char *peer, char *secret, int to)
{
	pc_event e;
	e.len = sizeof(pc_event);
	e.callno = callno;
	e.event = PC_EVENT_REGREQ;
	if (server)
		strncpy(e.e.regrequest.server, server, sizeof(e.e.regrequest.server)-1);
	else
		strcpy(e.e.regrequest.server, "");
	if (peer)
		strncpy(e.e.regrequest.peer, peer, sizeof(e.e.regrequest.peer)-1);
	else
		strcpy(e.e.regrequest.peer, "");
	if (secret)
		strncpy(e.e.regrequest.secret, secret, sizeof(e.e.regrequest.secret)-1);
	else
		strcpy(e.e.regrequest.secret, "");
	e.e.regrequest.refresh = to;
	return pc_write_event(src, &e);
}

struct simsock pc_get_fd(void)
{
	return *socks[SOURCE_GUI];
}

int pc_session_new(void)
{
	pc_event e;
	e.event = PC_EVENT_NEW;
	e.len = sizeof(pc_event);
	e.callno = 0;
	pc_write_event(SOURCE_GUI, &e);
	if (pc_read_event(SOURCE_PRI_OUT, &e))
		return -1;
	if (e.event != PC_EVENT_NEW_REPLY) {
		DEBU(G "Huh?  Got '%d' instead of new reply\n", e.event);
		return -1;
	}
	return e.e.newreply.id;
}

static int get_level(short *data, int samples)
{
	int sum=0;
	int x;
	for (x=0;x<samples;x++)
		sum += abs(data[x]);
	sum /= samples;
	return sum;
}

void calc_conf(struct peer *peer, short *data)
{
	int x;
	int y;
	int a;
	int s;
	short sum[CHUNKLEN] = { 0, };
	short *out;
	char fr[33];
	/* If we're in a direct talking situation, get out of it */
	if (!peer && (current > -1))
		return;
	/* Start by adding up anyone but us */
	for (x=0;x<PC_MAX_CALLS;x++) {
		if (peers[x] && (peers[x] != peer) && (peers[x]->inconf)) {
			a = peers[x]->cnfout;
			CNF_INC(a);
			if (a != peers[x]->cnfin) {
				/* If we have some data that we can conference, add it in */
				out = peers[x]->cnfbuf + CHUNKLEN * peers[x]->cnfout;
				for (y=0;y<CHUNKLEN;y++) {
					s = sum[y] + out[y];
					if (s > 32767)
						s = 32767;
					else if (s < -32768)
						s = -32768;
					sum[y] = s;
				}
			}
#if 0
			 else
				printf("Having to add in silence\n");
#endif
		}
	}
	if (peer) {
		if (current < 0) {
			/* We have a conference up and are talking to it, add in local speaker */
			for (y=0;y<CHUNKLEN;y++) {
				s = sum[y] + data[y];
				if (s > 32767)
					s = 32767;
				else if (s < -32768)
					s = -32768;
				sum[y] = s;
			}
		}
		/* Encode and transmit */
		if (!peer->gsmout)
			peer->gsmout = gsm_create();
		gsm_encode(peer->gsmout, sum, fr);
		iax_send_voice(peer->session, AST_FORMAT_GSM, fr, sizeof(fr));
	} else
		audio_send(audioc, AST_FORMAT_SLINEAR, sum, sizeof(sum), 0, 1);

}

int deliver_sound(int format, void *data, int len)
{
	pc_event e;
	int x;
	char fr[33];
	int a;

	EnterCriticalSection(&ready_lock);
	if (ready) {
		e.event = PC_EVENT_AUDIO;
		e.len = sizeof(pc_event);
		e.callno = -1;
		e.e.audio.level = get_level((short *)data, len/2);
		if (pc_write_event(SOURCE_PC, &e))
			DEBU(G "Unable to notify about sound event!\n");
		ready--;
		if (cnfcount) {
			/* Gotta calculate conference stuff for everyone */
			for (x=0;x<PC_MAX_CALLS;x++) {
				if (peers[x] && peers[x]->inconf)
					calc_conf(peers[x], (short *)data);
			}
			calc_conf(NULL, NULL);
			for (x=0;x<PC_MAX_CALLS;x++) {
				if (peers[x] && peers[x]->inconf) {
					/* Increment the input if possible */
					a = peers[x]->cnfout;
					CNF_INC(a);
					if (a != peers[x]->cnfin) {
						CNF_INC(peers[x]->cnfout);
					}
				}
			}
		} else {
			if ((current > -1) && peers[current] && !peers[current]->mute) {
				if (!peers[current]->gsmout)
					peers[current]->gsmout = gsm_create();
				gsm_encode(peers[current]->gsmout, data, fr);
				iax_send_voice(peers[current]->session, AST_FORMAT_GSM, fr, sizeof(fr));
			}
		}
	} else {
		//DEBU(G "Ready State = %d len= %d", ready, len);
    }
	LeaveCriticalSection(&ready_lock);
	return 0;
}

static int read_audio(int *id, int fd, short events, void *data)
{
	struct audio_channel *c = data;
	static char buffer[65536];
	static int sofar = 0;
	int bufferlen = chunk - sofar;
	if (events & AST_IO_PRI) {
		if (c && c->exception)
			c->exception(c);
		else
			DEBU(G "Exception and no handler for %s\n", c->name);
	}
	if (events & AST_IO_IN) {
		if (c->readaudio(c, format, buffer + sofar, &bufferlen)) {
			DEBU(G "Error reading voice data on %s\n", c->name);
			return 1;
		}
		sofar += bufferlen;
		if (sofar > chunk) {
			DEBU(G "Huh?  I have too much data\n");
			sofar = 0;
			return 1;
		}
		if ((sofar == chunk) || !enforcechunk) {
			deliver_sound(format, buffer, sofar);
			sofar = 0;
		}
	}
	return 1;
}

static int pc_audio_select(struct audio_channel *c)
{
	if (id)
		ast_io_remove(io, id);
	if (c->id > -1)
		ast_sched_del(sched, c->id);
	if (c->dpid > -1)
		ast_sched_del(sched, c->dpid);
	id = NULL;
	c->dpid = -1;
	c->id = -1;
	if (audioc)
		audioc->close(audioc);
	audioc = NULL;
	if (!c->open(c)) {
		audioc = c;
		if (c->fd > -1) {
			if (c->exception)
				id = ast_io_add(io, c->fd, read_audio, AST_IO_IN | AST_IO_PRI, c);
			else
				id = ast_io_add(io, c->fd, read_audio, AST_IO_IN, c);
		}
		return 0;
	}
	return -1;

}

static int toid = -1;

static int do_iax_timeout(void *data)
{
	struct iax_event *e;
	while((e = iax_get_event(0))) {
		handle_event(e);
		iax_event_free(e);
	}
	toid = -1;
	return 0;
}

void *phonecore_thread(void *unused)
{
	int ms;
	int ms2;
#ifdef MEASURE_TME
	int tme;
	int maxtme = 0;
	struct timeval last = { 0,0};
	struct timeval tv;
#endif
#ifdef USE_FORK
	int x;

	for (x=0;x<256;x++) {
		/* Close all file descriptors we don't need */
		if ((x != STDIN_FILENO) &&
			(x != STDOUT_FILENO) &&
			(x != STDERR_FILENO) &&
			(x != socks[SOURCE_PC]) &&
			(x != socks[SOURCE_PRI_IN]) &&
			(x != iax_get_fd()))
				close(x);
	}
#endif
	for (;;) {
		if (toid > -1)
			ast_sched_del(sched, toid);
		toid = -1;
		do {
			ms2 = iax_time_to_next_event();
			if (!ms2)
				do_iax_timeout(NULL);
		} while (!ms2);
		if (ms2 > -1)
			toid = ast_sched_add(sched, ms2, do_iax_timeout, NULL);
		ms = ast_sched_wait(sched);
#ifdef MEASURE_TME
		if (last.tv_sec || last.tv_usec) {
			gettimeofday(&tv, NULL);
			tme = (tv.tv_sec - last.tv_sec) * 1000 +
					(tv.tv_usec - last.tv_usec) / 1000;
		}
		if (tme > maxtme)
			maxtme = tme;
#endif
//		ast_io_wait(io, ms);
		if (ms<0)
			ms=10;
		ast_io_wait(io, ms);
//		sleep(ms);
//		ast_io_wait(io, ms);

#ifdef MEASURE_TME
		gettimeofday(&last, NULL);
#endif
		ast_sched_runq(sched);
	}
	/* Never reached */
	return NULL;
}

static int unset_duplex(void *av)
{
	struct audio_channel *ac = av;
	/* Turn it back to read-mode */
	printf("Unsetting duplex!\n");
	ac->simduplex(ac, 0);
	ac->dpid = -1;
	return 0;
}

static int audio_send(struct audio_channel *c, int format, void *data, int datalen, int loud, int stop)
{
	if (!c)
		return 0;
	if (stop) {
		if (c->id > -1)
			ast_sched_del(sched, c->id);
		c->id = -1;
	}
	if (!c->duplex) {
		/* If it's not loud, and we're still reading, keep reading.. */
		if (!loud && !c->writemode)
			return 0;
		/* Make sure we switch back to read mode if necessary */
		if (loud) {
			/* If this was "loud" then we reset the timer */
			if (c->dpid > -1)
				ast_sched_del(sched, c->dpid);
			c->dpid = ast_sched_add(sched, switchtime, unset_duplex, c);
		} else if (c->dpid < 0) /* Be sure there is a timer if quiet, but don't reset it */
			c->dpid = ast_sched_add(sched, switchtime, unset_duplex, c);

		if (!c->writemode)
			c->simduplex(c, 1);

	}
	return c->sendaudio(c, format, data, datalen);
}
