/*
 * Asterisk -- A telephony toolkit for Linux.
 * :folding=explicit:collapseFolds=1:
 *
 * PostgreSQL CDR Logger
 *
 * Ray Russell Reese III (russ@zerotech.net) 
 * Based on cdr_pgsql.c by James Sharp <jsharp@psychoses.org>
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License.
 *
 */

// {{{ Includes/Defines
#include <sys/types.h>
#include <asterisk/config.h>
#include <asterisk/options.h>
#include <asterisk/channel.h>
#include <asterisk/cdr.h>
#include <asterisk/module.h>
#include <asterisk/logger.h>
#include "../asterisk.h"

#include <stdio.h>
#include <string.h>

#include <stdlib.h>
#include <unistd.h>
#include <time.h>

#include <libpq-fe.h>

#define DATE_FORMAT "%Y-%m-%d %T"
// }}}

// {{{ Global Variables
static char *desc = "PostgreSQL CDR Backend";
static char *name = "pgsql";
static char *config = "cdr_pgsql.conf";

static char *table;
static char *dbhostname;
static char *dbport;
static char *dbname;
static char *dbuser;
static char *dbpassword;

static PGresult *res;
static PGconn *conn;
// }}}

// {{{ pgsql_log()
static int pgsql_log(struct ast_cdr *cdr)
{
	struct tm tm;
	struct timeval tv;
	struct timezone tz;
	char *sqlcmd, timestr[128];
	time_t t;
	
	sqlcmd = (char *)malloc(2048);
	memset(sqlcmd,0,2048);
	
	gettimeofday(&tv,&tz);
	t = tv.tv_sec;
	localtime_r(&t,&tm);
	strftime(timestr,128,DATE_FORMAT,&tm);
	
	
	ast_log(LOG_DEBUG,"cdr_pgsql: inserting a CDR record.\n");
	sprintf(
		sqlcmd,
		"INSERT INTO %s (calldate, clid, src, dst, dcontext, channel, dstchannel, lastapp, lastdata, duration, billsec, disposition, amaflags, accountcode) VALUES ('%s','%s','%s','%s','%s', '%s','%s','%s','%s',%i,%i,%i,%i,'%s')",
		table,
		timestr,
		cdr->clid,
		cdr->src,
		cdr->dst,
		cdr->dcontext,
		cdr->channel,
		cdr->dstchannel,
		cdr->lastapp,
		cdr->lastdata,
		cdr->duration,
		cdr->billsec,
		cdr->disposition,
		cdr->amaflags,
		cdr->accountcode
	);
	
	ast_log(LOG_DEBUG, "cdr_pgsql: SQL command as follows:  %s\n", sqlcmd);
	
	res = PQexec(conn, sqlcmd);
	
	if (PQresultStatus(res) != PGRES_COMMAND_OK) {
		ast_log(LOG_ERROR, "Failed to insert into PostgreSQL database.\n");
		free(sqlcmd);
		PQclear(res);
		return -1;
	}

	PQclear(res);		
	free(sqlcmd);
	
	return 0;
}
// }}}

// {{{ description()
char *description(void)
{
	return desc;
}
// }}}

// {{{ unload_module()
int unload_module(void)
{ 
	PQfinish(conn);
	ast_cdr_unregister(name);

	return 0;
}
// }}}

// {{{ load_config()
int load_config()
{
	struct ast_config *cfg;
	struct ast_variable *var;

	cfg = ast_load(config);
	if (!cfg) {
		ast_log(LOG_WARNING, "Unable to load config for pgsql CDR's: %s\n", config);
		return 0;
	}
	
	var = ast_variable_browse(cfg, "global");
	if (!var) {
		/* nothing configured */
		return 0;
	}

	dbhostname = ast_variable_retrieve(cfg, "global","hostname");
	dbname = ast_variable_retrieve(cfg, "global","dbname");
	dbuser = ast_variable_retrieve(cfg, "global","user");
	dbpassword = ast_variable_retrieve(cfg, "global","password");
	dbport = ast_variable_retrieve(cfg, "global", "port");
	table = ast_variable_retrieve(cfg, "global", "table");
	
	ast_log(LOG_DEBUG,"cdr_pgsql: got hostname of %s\n", dbhostname);
	ast_log(LOG_DEBUG,"cdr_pgsql: got user of %s\n", dbuser);
	ast_log(LOG_DEBUG,"cdr_pgsql: got dbname of %s\n", dbname);
	ast_log(LOG_DEBUG,"cdr_pgsql: got port of %s\n", dbport);
	ast_log(LOG_DEBUG,"cdr_pgsql: got password of %s\n", dbpassword);
	ast_log(LOG_DEBUG,"cdr_pgsql: got table of %s\n", table);

	/**
		We only care about making sure a table if specified,
		the rest can be NULL to cause PostgreSQL to use the default
		client connection options.
	*/
	
	if (table == NULL) {
		ast_log(LOG_ERROR, "CDR Table not specified.\n");
		return -1;
	}
}
// }}}

// {{{ connectdb()
int connectdb()
{
	conn = PQsetdbLogin(
		dbhostname,
		dbport,
		NULL,
		NULL,
		dbname,
		dbuser,
		dbpassword
	);
	
	if (PQstatus(conn) == CONNECTION_BAD) {
		ast_log(LOG_ERROR, "Connection to PostgreSQL failed.\n");
		return -1;
	} else {
		ast_log(LOG_DEBUG, "Connection to PostgreSQL successfull.\n");
	}
}
// }}}

// {{{ load_module()
int load_module(void)
{
	int res;
	
	if (load_config() == -1) {
	  return -1;
	}
	
	if (connectdb() == -1) {
	  return -1;
	}
		
	res = ast_cdr_register(name, desc, pgsql_log);

	if (res) {
		ast_log(LOG_ERROR, "Unable to register PostgreSQL CDR handling.\n");
	}

	return res;
}

// {{{ reload()
int reload(void)
{
	
	PQfinish(conn);
	
	if (load_config() == -1) {
	  return -1;
	}
	
	if (connectdb() == -1) {
	  return -1;
	}
	
	return 0;
}
// }}}

// {{{ usecount()
int usecount(void)
{
	return 0;
}
// }}}

// {{{ key()
char *key()
{
	return ASTERISK_GPL_KEY;
}
// }}}
